#!/usr/bin/perl
#
# Perl client for RFB (VNC) protocol
#
# This script sits between a VNC client and a VNC server and rotates the display 180 degrees

use Socket;

use FileHandle;
use IPC::Open2;

# For speed, we use a C routine to actually flip the bytes in 'raw' encoding
# The 'use lib' lines are there in case we haven't installed the extension (yet).

use lib "byteflip/blib/lib";
use lib "byteflip/blib/arch/auto/byteflip";
use byteflip;

$debug = 1;

# This is the server we're connecting to

$remote = 'localhost';
$remotedisplayno = 3;

# This is the psuedo-server number we'll create on the local host

$localdisplayno = 2;

# True if we're doing screen rotation

$rotate180 = 0;

open(LOGFILE, ">logfile");
select LOGFILE;
$| = 1;

sub spawn;
#sub logmsg { print "$0 $$: @_ at ", scalar localtime, "\n" if $debug }
sub logmsg { print LOGFILE "$0 $$ at ", scalar localtime, ": @_\n" if $debug }
#sub logmsg { print "$0 $$: @_ \n" if $debug }

sub decode_pixfmt {
    my ($pixfmt) = @_;

    ($bitsperpixel, $depth, $bigendian, $truecolour,
     $redmax, $greenmax, $bluemax, $redshift, $greenshift, $blueshift)
	= unpack('CCCCnnnCCCxxx', $pixfmt);

    # global variable we'll need to refer to later
    $bytesperpixel = $bitsperpixel/8;

    logmsg "$bitsperpixel bits per pixel; depth $depth";

}

# 0. listen for connections

$port = 5900 + $localdisplayno;

$paddr = sockaddr_in($port, INADDR_ANY);

$proto = getprotobyname('tcp');
socket(LISTENER, PF_INET, SOCK_STREAM, $proto)  || die "socket: $!";
bind(LISTENER, $paddr)  || die "bind: $!";
listen(LISTENER, 5);

my $waitedpid = 0;
my $paddr;

use POSIX ":sys_wait_h";
sub REAPER {
    my $child;
    while (($waitedpid = waitpid(-1,WNOHANG)) > 0) {
	logmsg "reaped $waitedpid" . ($? ? " with exit $?" : '');
    }
    $SIG{CHLD} = \&REAPER;  # loathe sysV
}

$SIG{CHLD} = \&REAPER;

for ( $waitedpid = 0; ($paddr = accept(CLIENT,LISTENER)) || $waitedpid; $waitedpid = 0, close CLIENT)
{
    next if $waitedpid and not $paddr;
    my($port,$iaddr) = sockaddr_in($paddr);
    my $name = gethostbyaddr($iaddr,AF_INET);

    logmsg "connection from $name [", inet_ntoa($iaddr), "] at port $port";

    spawn sub {

	# 1. connect to server

	$port = 5900 + $remotedisplayno;

	$iaddr = inet_aton($remote)   || die "no host: $remote";
	$paddr = sockaddr_in($port, $iaddr);

	$proto = getprotobyname('tcp');
	socket(SERVER, PF_INET, SOCK_STREAM, $proto)  || die "socket: $!";
	connect(SERVER, $paddr)  || die "connect: $!";

	select SERVER;
	$| = 1;
	select CLIENT;
	$| = 1;
	select STDERR;

	# 2. server sends client a ProtocolVersion message

	read(SERVER, $ProtocolVersionMsg, 12) || die "reading ProtocolVersion: $!";
	logmsg "Server ProtocolVersion: $ProtocolVersionMsg";
	print CLIENT $ProtocolVersionMsg;

	# 3. client sends server a ProtocolVersion message

	read(CLIENT, $ProtocolVersionMsg, 12) || die "reading ProtocolVersion: $!";
	logmsg "Client ProtocolVersion: $ProtocolVersionMsg";
	print SERVER $ProtocolVersionMsg;

	# 4. server sends authentication scheme

	read(SERVER, $AuthenticationSchemeMsg, 4) || die "reading AuthenticationScheme: $!";
	print CLIENT $AuthenticationSchemeMsg;
	$AuthenticationScheme = unpack('N', $AuthenticationSchemeMsg);
	if ($AuthenticationScheme == 0) {
	    logmsg "AuthenticationScheme: connection failed (0)";

	    # read the failure reason

	    read(SERVER, $ReasonLengthMsg, 4) || die "reading ReasonLength: $!";
	    print CLIENT $ReasonLengthMsg;
	    $ReasonLength = unpack('N', $ReasonLengthMsg);
	    read(SERVER, $Reason, $ReasonLength) || die "reading Reason: $!";
	    print CLIENT $Reason;

	    logmsg "Failure Reason: $Reason";
	    exit;

	} elsif ($AuthenticationScheme == 1) {
	    logmsg "AuthenticationScheme: no authentication (1)";
	} elsif ($AuthenticationScheme == 2) {
	    logmsg "AuthenticationScheme: VNC authentication (2)";

	    read(SERVER, $Challenge, 16) || die "reading Challenge: $!";
	    print CLIENT $Challenge;

	    read(CLIENT, $Response, 16) || die "reading Response: $!";
	    print SERVER $Response;

	    read(SERVER, $AuthenticationResponseMsg, 4) || die "reading AuthenticationResponse: $!";
	    print CLIENT $AuthenticationResponseMsg;

	    $AuthenticationResponse = unpack('N', $AuthenticationResponseMsg);

	    if ($AuthenticationResponse == 1) {
		logmsg "VNC Authentication Failed";
		exit;
	    }
	}

	# 5. Client sends Client Initialization

	read(CLIENT, $ClientInitialization, 1);
	print SERVER $ClientInitialization;

	# 6. Server sends Server Initialization

	read(SERVER, $ServerInitialization, 24);
	print CLIENT $ServerInitialization;
	($screen_width, $screen_height, $pixfmt, $namelen) = unpack('nna16N', $ServerInitialization);
	&decode_pixfmt($pixfmt);
	read(SERVER, $NameString, $namelen);
	print CLIENT $NameString;

	# 7. Main Loop

	$rin = $win = $ein = '';
	vec($rin,fileno(SERVER),1) = 1;
	vec($rin,fileno(CLIENT),1) = 1;
	$ein = $rin | $win;

	while (1) {

	    $nfound = select($rout=$rin, $wout=$win, $eout=$ein, undef);

	    if ((vec($eout,fileno(CLIENT),1) == 1) || (vec($eout,fileno(SERVER),1) == 1)) {
		close(CLIENT);
		close(SERVER);
		exit;
	    }


	    if (vec($rout,fileno(CLIENT),1) == 1) {
		# Client message
		if (read(CLIENT, $ClientMsgType, 1) == 0) {
		    close(SERVER);
		    exit;
		}
		print SERVER $ClientMsgType;

		$msgnum = unpack('C', $ClientMsgType);

		if ($msgnum == 0) {

		    logmsg "Client message 0 - SetPixelFormat";

		    read(CLIENT, $SetPixelFormat, 19);
		    print SERVER $SetPixelFormat;
		    $pixfmt = unpack('x3a16', $SetPixelFormat);
		    &decode_pixfmt($pixfmt);

		} elsif ($msgnum == 1) {

		    logmsg "Client message 1 - FixColourMapEntries";

		    read(CLIENT, $FixColourMapEntries, 5);
		    print SERVER $FixColourMapEntries;
		    $numcolours = unpack('x3n', $FixColourMapEntries);
		    read(CLIENT, $ColourMap, 6*$numcolours);
		    print SERVER $ColourMap;

		} elsif ($msgnum == 2) {

		    logmsg "Client message 2 - SetEncodings";

		    read(CLIENT, $SetEncodings, 3);
		    $numcodings = unpack('xn', $SetEncodings);
		    read(CLIENT, $Codings, 4*$numcodings);
		    @Codings = unpack("N[$numcodings]",$Codings);

		    logmsg "   old $numcodings @Codings";

		    # edit out of the list of codings any this script doesn't understand
		    # mainly that's 5 (hextile) and 16 (ZRLE)

		    @Codings = map { (($_ <= 4) and ($_ != 3)) ? $_ : () } @Codings;
		    $numcodings = $#Codings + 1;
		    $SetEncodings = pack('xn', $numcodings);
		    $Codings = pack("N[$numcodings]",@Codings);

		    logmsg "   new $numcodings @Codings";

		    print SERVER $SetEncodings;
		    print SERVER $Codings;

		} elsif ($msgnum == 3) {

		    logmsg "Client message 3 - FramebufferUpdateRequest";

		    read(CLIENT, $FramebufferUpdateRequest, 9);

		    if ($rotate180) {
			($incremental, $x, $y, $width, $height) = unpack('Cn4', $FramebufferUpdateRequest);
			logmsg "  old incremental=$incremental, x=$x, y=$y, width=$width, height=$height";
			$x = $screen_width - $x - $width;
			$y = $screen_height - $y - $height;
			logmsg "  new incremental=$incremental, x=$x, y=$y, width=$width, height=$height";
			$FramebufferUpdateRequest = pack('Cn4', $incremental, $x, $y, $width, $height);
		    }

		    print SERVER $FramebufferUpdateRequest;

		} elsif ($msgnum == 4) {

		    logmsg "Client message 4 - KeyEvent";

		    read(CLIENT, $KeyEvent, 7);
		    print SERVER $KeyEvent;

		} elsif ($msgnum == 5) {

		    logmsg "Client message 5 - PointerEvent";

		    read(CLIENT, $PointerEvent, 5);

		    if ($rotate180) {
			($buttons, $x, $y) = unpack('Cn2', $PointerEvent);
			$x = $screen_width - $x - 1;
			$y = $screen_height - $y - 1;
			$PointerEvent = pack('Cn2', $buttons, $x, $y);
		    }

		    print SERVER $PointerEvent;

		} elsif ($msgnum == 6) {

		    logmsg "Client message 6 - ClientCutText";

		    read(CLIENT, $ClientCutText, 7);
		    print SERVER $ClientCutText;
		    $strlen = unpack('x3N', $ClientCutText);
		    read(CLIENT, $CutText, $strlen);
		    print SERVER $CutText;

		} else {

		    logmsg "Client message $msgnum - UNKNOWN!!";

		}
	    }

	    if (vec($rout,fileno(SERVER),1) == 1) {
		# Server message
		if (read(SERVER, $ServerMsgType, 1) == 0) {
		    close(CLIENT);
		    exit;
		}
		print CLIENT $ServerMsgType;

		$msgnum = unpack('C', $ServerMsgType);

		if ($msgnum == 0) {

		    logmsg "Server message 0 - FramebufferUpdate";

		    read(SERVER, $FramebufferUpdate, 3);
		    print CLIENT $FramebufferUpdate;
		    $numrects = unpack('xn', $FramebufferUpdate);

		    logmsg "   $numrects rectangles";

		    for ($i=0; $i<$numrects; $i++) {

			read(SERVER, $FramebufferUpdateRect, 12);
			($x, $y, $width, $height, $encoding) = unpack('n4N', $FramebufferUpdateRect);

			logmsg "   rect $i: x=$x y=$y width=$width height=$height encoding=$encoding";

			if ($rotate180) {
			    $x = $screen_width - $x - $width;
			    $y = $screen_height - $y - $height;
			    $FramebufferUpdateRect = pack('n4N', $x, $y, $width, $height, $encoding);
			}

			print CLIENT $FramebufferUpdateRect;

			if ($encoding == 0) {

			    # raw encoding

			    # the actual rotation of the pixels is done in C for speed (see byteflip.xs)
			    # byteflip::flip is just this:
			    #
			    # for (i=0; i<len/2; i+=num) {
			    #    for (j=0; j<num; j++) {
			    #       char c=str[i+j];
			    #       str[i+j] = str[len-i-num+j];
			    #       str[len-i-num+j] = c;
			    #    }
			    # }

			    read(SERVER, $Pixdata, $width*$height*$bytesperpixel);
			    if ($rotate180) {
			      byteflip::flip($Pixdata, $width*$height*$bytesperpixel, $bytesperpixel);
			    }
			    print CLIENT $Pixdata;

			} elsif ($encoding == 1) {

			    # copy rectangle
			    read(SERVER, $Rect, 4);
			    if ($rotate180) {
				($srcx, $srcy) = unpack('n2', $Rect);
				$srcx = $screen_width - $srcx - 1;
				$srcy = $screen_height - $srcy - 1;
				$Rect = pack('n2', $srcx, $srcy);
			    }
			    print CLIENT $Rect;

			} elsif ($encoding == 2) {

			    # RRE - rectangle sent as a background value then a series of single-color subrectangles

			    read(SERVER, $RREheader, 4+$bytesperpixel);
			    ($numsubrects, $backgroundpixel) = unpack("Na[$bytesperpixel]", $RREheader);
			    print CLIENT $RREheader;
			    for ($j=0; $j<$numsubrects; $j++) {
				read(SERVER, $RREsubrect, 8+$bytesperpixel);
				($pixelval, $subx, $suby, $subw, $subh) = unpack("a[$bytesperpixel]n4", $RREsubrect);
				logmsg "      subrect $j/$numsubrects: x=$subx y=$suby w=$subw h=$subh";
				if ($rotate180) {
				    $subx = $width - $subx - $subw;
				    $suby = $height - $suby - $subh;
				    $RREsubrect = pack("a[$bytesperpixel]n4", $pixelval, $subx, $suby, $subw, $subh);
				}
				print CLIENT $RREsubrect;
			    }

			} elsif ($encoding == 4) {

			    # CoRRE - basically the same as RRE, except that we use 1-byte values instead of 2-byte
			    # values for our coordinates.  So in theory, we could get into a situation where the server
			    # sent a rectangle larger than 255x255 because its subrectangle coordinates all fit in one
			    # byte each, but after rotation no longer do so.  In practice, this never happens, partially
			    # because it's a bizarre special case and partially because the X server, at least, never
			    # sends CoRRE rectangles larger than 48x48.

			    read(SERVER, $RREheader, 4+$bytesperpixel);
			    ($numsubrects, $backgroundpixel) = unpack("Na[$bytesperpixel]", $RREheader);
			    print CLIENT $RREheader;

			    for ($j=0; $j<$numsubrects; $j++) {
				read(SERVER, $RREsubrect, 4+$bytesperpixel);
				($pixelval, $subx, $suby, $subw, $subh) = unpack("a[$bytesperpixel]C4", $RREsubrect);
				logmsg "      subrect $j/$numsubrects: x=$subx y=$suby w=$subw h=$subh";
				if ($rotate180) {
				    $subx = $width - $subx - $subw;
				    $suby = $height - $suby - $subh;
				    $RREsubrect = pack("a[$bytesperpixel]C4", $pixelval, $subx, $suby, $subw, $subh);
				}
				print CLIENT $RREsubrect;
			    }

			} elsif ($encoding == 5) {

			    # hextile - because the tiles are encoding left-to-right, top-to-bottom, there's no way to
			    # rotate a hextile-encoding rectangle with reading the whole thing in first.  Furthermore,
			    # since the hextiles along the right or bottom edges are truncated if the width or height
			    # isn't a multiple of 16, the rotated hextiles might not correspond one-to-one with the
			    # original hextiles.  Right now, I have a simple way of dealing with this problem - I don't
			    # support hextile encoding!  The perl here can at least decode the hextile, but back
			    # up in SetEncodings I made sure that we never request hextile from the server.

			    $numtiles = int(($width+15)/16)*int(($height+15)/16);
			    for ($j=0; $j<$numtiles; $j++) {
				read(SERVER, $HEXtileEncoding, 1);
				print CLIENT $HEXtileEncoding;
				$subencoding = unpack('C', $HEXtileEncoding);
				logmsg "      subrect $j/$numtiles: subencoding=$subencoding";
				if ($subencoding & 1) {
				    # raw subencoding - hard part is figuring out if tile is on the right or bottom edge
				    if (($j+1)%int(($width+15)/16) == 0) {
					$subw = $width % 16;
				    } else {
					$subw = 16;
				    }
				    if ($j >= int(($width+15)*16)*int(($height-1)/16)) {
					$subh = $height % 16;
				    } else {
					$subh = 16;
				    }
				    read(SERVER, $Pixdata, $subw*$subh);
				    print CLIENT $Pixdata;
				} else {
				    # compressed subencoding
				    if ($subencoding & 2) {
					read(SERVER, $HEXtileBackground, $bytesperpixel);
					print CLIENT $HEXtileBackground;
				    }
				    if ($subencoding & 4) {
					read(SERVER, $HEXtileForeground, $bytesperpixel);
					print CLIENT $HEXtileForeground;
				    }
				    if ($subencoding & 8) {
					read(SERVER, $HEXtileSubrects, 1);
					print CLIENT $HEXtileSubrects;
					$subrects = unpack('C', $HEXtileSubrects);
					for ($k=0; $k<$subrects; $k++) {
					    if ($subencoding & 16) {
						read(SERVER, $HEXtilePixval, $bytesperpixel);
						print CLIENT $HEXtilePixval;
					    }
					    read(SERVER, $HEXtileSubrect, 2);
					    ($subxy, $subwh) = unpack('CC', $HEXtileSubrect);
					    $subx = $subxy >> 4;
					    $suby = $subxy & 15;
					    $subw = ($subwh >> 4) + 1;
					    $subh = ($subwh & 15) + 1;
					    print CLIENT $HEXtileSubrect;
					}
				    }
				}
			    }
			}
		    }

		} elsif ($msgnum == 1) {

		    logmsg "Server message 1 - SetColourMapEntries";

		    read(SERVER, $SetColourMapEntries, 5);
		    print CLIENT $SetColourMapEntries;
		    $numcolours = unpack('x3n', $SetColourMapEntries);
		    read(SERVER, $ColourMap, 6*$numcolours);
		    print CLIENT $ColourMap;

		} elsif ($msgnum == 2) {

		    logmsg "Server message 2 - Bell";

		} elsif ($msgnum == 3) {

		    logmsg "Server message 3 - ServerCutText";

		    read(SERVER, $ServerCutText, 7);
		    print CLIENT $ServerCutText;
		    $strlen = unpack('x3N', $ServerCutText);
		    read(SERVER, $CutText, $strlen);
		    print CLIENT $CutText;

		} else {

		    logmsg "Server message $msgnum - UNKNOWN!!";

		}
	    }

	}


    };
}


# A subroutine ripped from the perl documentation to spawn a subroutine in a child process

sub spawn {
    my $coderef = shift;

    unless (@_ == 0 && $coderef && ref($coderef) eq 'CODE') {
	die "usage: spawn CODEREF";
    }

    my $pid;
    if (!defined($pid = fork)) {
	logmsg "cannot fork: $!";
	return;
    } elsif ($pid) {
	logmsg "begat $pid";
	return; # I'm the parent
    }
               # else I'm the child -- go spawn

    open(STDIN,  "<&CLIENT")   || die "can't dup client to stdin";
    open(STDOUT, ">&CLIENT")   || die "can't dup client to stdout";
               ## open(STDERR, ">&STDOUT") || die "can't dup stdout to stderr";
    exit &$coderef();
}
